<?php
/**
 * Main plugin class
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_SSL_Manager {
    
    private $api;
    private $admin;
    private $cpanel;
    
    public function __construct() {
        $this->api = new WP_SSL_API();
        $this->admin = new WP_SSL_Admin();
        $this->cpanel = new WP_SSL_CPanel();
    }
    
    public function run() {
        // Hook into WordPress
        add_action('admin_menu', array($this->admin, 'add_admin_menu'));
        add_action('admin_init', array($this->admin, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_wp_ssl_generate_certificate', array($this, 'ajax_generate_certificate'));
        add_action('wp_ajax_wp_ssl_verify_domain', array($this, 'ajax_verify_domain'));
        add_action('wp_ajax_wp_ssl_download_certificate', array($this, 'ajax_download_certificate'));
        add_action('wp_ajax_wp_ssl_install_certificate', array($this, 'ajax_install_certificate'));
        add_action('wp_ajax_wp_ssl_get_certificates', array($this, 'ajax_get_certificates'));
    }
    
    public function enqueue_scripts($hook) {
        if (strpos($hook, 'wp-ssl-manager') === false) {
            return;
        }
        
        wp_enqueue_script('wp-ssl-manager-js', WP_SSL_MANAGER_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), WP_SSL_MANAGER_VERSION, true);
        wp_enqueue_style('wp-ssl-manager-css', WP_SSL_MANAGER_PLUGIN_URL . 'assets/css/admin.css', array(), WP_SSL_MANAGER_VERSION);
        
        wp_localize_script('wp-ssl-manager-js', 'wpSslManager', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wp_ssl_manager_nonce'),
            'strings' => array(
                'generating' => __('Generating certificate...', 'wp-ssl-manager'),
                'verifying' => __('Verifying domain...', 'wp-ssl-manager'),
                'downloading' => __('Downloading certificate...', 'wp-ssl-manager'),
                'installing' => __('Installing certificate...', 'wp-ssl-manager'),
                'success' => __('Operation completed successfully!', 'wp-ssl-manager'),
                'error' => __('An error occurred. Please try again.', 'wp-ssl-manager'),
            ),
        ));
    }
    
    public function ajax_generate_certificate() {
        check_ajax_referer('wp_ssl_manager_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'wp-ssl-manager')));
        }
        
        $domain = sanitize_text_field($_POST['domain']);
        $san_domains = isset($_POST['san_domains']) ? array_map('sanitize_text_field', $_POST['san_domains']) : array();
        $verification_method = sanitize_text_field($_POST['verification_method']);
        $provider = sanitize_text_field($_POST['provider']);
        
        $result = $this->api->create_certificate($domain, $san_domains, $verification_method, $provider);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_verify_domain() {
        check_ajax_referer('wp_ssl_manager_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'wp-ssl-manager')));
        }
        
        $certificate_id = intval($_POST['certificate_id']);
        $verification_id = intval($_POST['verification_id']);
        
        // Create verification file if HTTP method
        $verification = $this->api->get_verification($certificate_id, $verification_id);
        
        if ($verification && $verification['method'] === 'http-01') {
            $this->create_verification_file($verification);
        }
        
        // Verify domain
        $result = $this->api->verify_domain($certificate_id, $verification_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_download_certificate() {
        check_ajax_referer('wp_ssl_manager_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'wp-ssl-manager')));
        }
        
        $certificate_id = intval($_POST['certificate_id']);
        $format = sanitize_text_field($_POST['format']);
        
        $result = $this->api->download_certificate($certificate_id, $format);
        
        if ($result['success']) {
            // Save certificate files locally
            $upload_dir = wp_upload_dir();
            $ssl_dir = $upload_dir['basedir'] . '/ssl-certificates';
            
            if (!file_exists($ssl_dir)) {
                wp_mkdir_p($ssl_dir);
            }
            
            $certificate_data = base64_decode($result['content']);
            
            if ($result['content_type'] === 'application/zip') {
                $zip_file = $ssl_dir . '/' . $result['filename'];
                file_put_contents($zip_file, $certificate_data);
                
                // Extract ZIP
                $zip = new ZipArchive();
                if ($zip->open($zip_file) === TRUE) {
                    $zip->extractTo($ssl_dir);
                    $zip->close();
                    unlink($zip_file);
                }
                
                wp_send_json_success(array(
                    'message' => __('Certificate downloaded and extracted successfully', 'wp-ssl-manager'),
                    'path' => $ssl_dir,
                ));
            } else {
                $file_path = $ssl_dir . '/' . $result['filename'];
                file_put_contents($file_path, $certificate_data);
                
                wp_send_json_success(array(
                    'message' => __('Certificate downloaded successfully', 'wp-ssl-manager'),
                    'path' => $file_path,
                ));
            }
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_install_certificate() {
        check_ajax_referer('wp_ssl_manager_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'wp-ssl-manager')));
        }
        
        $certificate_id = intval($_POST['certificate_id']);
        $format = sanitize_text_field($_POST['format']);
        
        // First download the certificate
        $download_result = $this->api->download_certificate($certificate_id, $format);
        
        if (!$download_result['success']) {
            wp_send_json_error($download_result);
        }
        
        // Save certificate files
        $upload_dir = wp_upload_dir();
        $ssl_dir = $upload_dir['basedir'] . '/ssl-certificates';
        
        if (!file_exists($ssl_dir)) {
            wp_mkdir_p($ssl_dir);
        }
        
        $certificate_data = base64_decode($download_result['content']);
        $domain = sanitize_text_field($_POST['domain']);
        
        // Handle ZIP or single file
        if ($download_result['content_type'] === 'application/zip') {
            $zip_file = $ssl_dir . '/temp-' . time() . '.zip';
            file_put_contents($zip_file, $certificate_data);
            
            $zip = new ZipArchive();
            if ($zip->open($zip_file) === TRUE) {
                $zip->extractTo($ssl_dir);
                $zip->close();
                unlink($zip_file);
            }
        } else {
            $file_path = $ssl_dir . '/' . $download_result['filename'];
            file_put_contents($file_path, $certificate_data);
        }
        
        // Install via cPanel
        $cpanel_host = get_option('wp_ssl_cpanel_host');
        $cpanel_user = get_option('wp_ssl_cpanel_user');
        $cpanel_pass = get_option('wp_ssl_cpanel_pass');
        $cpanel_port = get_option('wp_ssl_cpanel_port', '2083');
        
        if ($cpanel_host && $cpanel_user && $cpanel_pass) {
            $install_result = $this->cpanel->install_ssl($domain, $ssl_dir, $cpanel_host, $cpanel_user, $cpanel_pass, $cpanel_port);
            
            if ($install_result['success']) {
                wp_send_json_success(array(
                    'message' => __('SSL certificate installed successfully via cPanel', 'wp-ssl-manager'),
                ));
            } else {
                wp_send_json_error($install_result);
            }
        } else {
            wp_send_json_error(array('message' => __('cPanel credentials not configured', 'wp-ssl-manager')));
        }
    }
    
    public function ajax_get_certificates() {
        check_ajax_referer('wp_ssl_manager_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'wp-ssl-manager')));
        }
        
        $result = $this->api->get_certificates();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    private function create_verification_file($verification) {
        $file_path = $verification['file_path'];
        $file_content = $verification['file_content'];
        
        // Get WordPress uploads directory
        $upload_dir = wp_upload_dir();
        $verification_dir = $upload_dir['basedir'] . '/' . dirname($file_path);
        
        // Create directory if it doesn't exist
        if (!file_exists($verification_dir)) {
            wp_mkdir_p($verification_dir);
        }
        
        // Create verification file
        $full_path = $upload_dir['basedir'] . '/' . $file_path;
        file_put_contents($full_path, $file_content);
        
        // Also create in WordPress root .well-known directory
        $wp_verification_dir = ABSPATH . dirname($file_path);
        if (!file_exists($wp_verification_dir)) {
            wp_mkdir_p($wp_verification_dir);
        }
        file_put_contents(ABSPATH . $file_path, $file_content);
        
        return true;
    }
}

