jQuery(document).ready(function($) {
    'use strict';
    
    // Tab switching
    $('.nav-tab').on('click', function(e) {
        e.preventDefault();
        var target = $(this).attr('href');
        
        $('.nav-tab').removeClass('nav-tab-active');
        $(this).addClass('nav-tab-active');
        
        $('.tab-content').removeClass('active');
        $(target).addClass('active');
    });
    
    // Add SAN domain input
    $('#add-san-domain').on('click', function() {
        var input = $('<input type="text" name="san_domains[]" class="regular-text san-domain-input" placeholder="www.example.com">');
        $('#san-domains-container').append(input);
    });
    
    // Generate certificate form
    $('#wp-ssl-generate-form').on('submit', function(e) {
        e.preventDefault();
        
        var $form = $(this);
        var $result = $('#wp-ssl-generate-result');
        var $submit = $form.find('button[type="submit"]');
        
        $submit.prop('disabled', true).text(wpSslManager.strings.generating);
        $result.hide().removeClass('notice-success notice-error');
        
        var formData = {
            action: 'wp_ssl_generate_certificate',
            nonce: wpSslManager.nonce,
            domain: $('#domain').val(),
            san_domains: $('.san-domain-input').map(function() {
                return $(this).val();
            }).get().filter(function(val) {
                return val.length > 0;
            }),
            verification_method: $('#verification_method').val(),
            provider: $('#provider').val(),
        };
        
        $.ajax({
            url: wpSslManager.ajaxUrl,
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    $result.addClass('notice notice-success').html(
                        '<p><strong>' + wpSslManager.strings.success + '</strong></p>' +
                        '<p>Certificate ID: ' + response.data.certificate_id + '</p>' +
                        '<p>Status: ' + response.data.data.status + '</p>' +
                        '<div id="verification-steps"></div>'
                    ).show();
                    
                    // Show verification steps
                    if (response.data.data.verifications && response.data.data.verifications.length > 0) {
                        showVerificationSteps(response.data.data.id, response.data.data.verifications);
                    }
                    
                    // Refresh certificates list
                    loadCertificates();
                } else {
                    $result.addClass('notice notice-error').html(
                        '<p><strong>Error:</strong> ' + (response.data.message || wpSslManager.strings.error) + '</p>'
                    ).show();
                }
            },
            error: function() {
                $result.addClass('notice notice-error').html(
                    '<p><strong>Error:</strong> ' + wpSslManager.strings.error + '</p>'
                ).show();
            },
            complete: function() {
                $submit.prop('disabled', false).text('Generate Certificate');
            }
        });
    });
    
    // Show verification steps
    function showVerificationSteps(certId, verifications) {
        var html = '<h4>Verification Steps:</h4>';
        
        verifications.forEach(function(verification) {
            html += '<div class="verification-step" data-verification-id="' + verification.id + '">';
            html += '<p><strong>Domain:</strong> ' + verification.domain + '</p>';
            
            if (verification.method === 'http-01') {
                html += '<p><strong>File Path:</strong> <code>' + verification.file_path + '</code></p>';
                html += '<p><strong>File Content:</strong> <code>' + verification.file_content + '</code></p>';
                html += '<p class="description">The verification file has been created automatically.</p>';
            } else {
                html += '<p><strong>DNS Record Name:</strong> <code>' + verification.dns_record_name + '</code></p>';
                html += '<p><strong>DNS Record Value:</strong> <code>' + verification.dns_record_value + '</code></p>';
            }
            
            html += '<button class="button verify-domain-btn" data-cert-id="' + certId + '" data-verification-id="' + verification.id + '">Verify Domain</button>';
            html += '</div><hr>';
        });
        
        $('#verification-steps').html(html);
    }
    
    // Verify domain
    $(document).on('click', '.verify-domain-btn', function() {
        var $btn = $(this);
        var certId = $btn.data('cert-id');
        var verificationId = $btn.data('verification-id');
        
        $btn.prop('disabled', true).text(wpSslManager.strings.verifying);
        
        $.ajax({
            url: wpSslManager.ajaxUrl,
            type: 'POST',
            data: {
                action: 'wp_ssl_verify_domain',
                nonce: wpSslManager.nonce,
                certificate_id: certId,
                verification_id: verificationId,
            },
            success: function(response) {
                if (response.success) {
                    $btn.closest('.verification-step').html(
                        '<p class="notice notice-success"><strong>Verified!</strong> ' + response.data.message + '</p>'
                    );
                    
                    // Check if all verified
                    if (response.data.all_verified) {
                        $('#verification-steps').append(
                            '<p class="notice notice-success"><strong>All domains verified! Certificate is being issued...</strong></p>'
                        );
                        setTimeout(function() {
                            loadCertificates();
                        }, 2000);
                    }
                } else {
                    $btn.closest('.verification-step').append(
                        '<p class="notice notice-error">' + (response.data.message || 'Verification failed') + '</p>'
                    );
                    $btn.prop('disabled', false).text('Verify Domain');
                }
            },
            error: function() {
                $btn.prop('disabled', false).text('Verify Domain');
                alert(wpSslManager.strings.error);
            }
        });
    });
    
    // Load certificates
    function loadCertificates() {
        $('#certificates-list').html('<p>Loading certificates...</p>');
        
        $.ajax({
            url: wpSslManager.ajaxUrl,
            type: 'POST',
            data: {
                action: 'wp_ssl_get_certificates',
                nonce: wpSslManager.nonce,
            },
            success: function(response) {
                if (response.success && response.data.data && response.data.data.length > 0) {
                    var html = '';
                    response.data.data.forEach(function(cert) {
                        html += '<div class="certificate-item" data-id="' + cert.id + '">';
                        html += '<h3>' + cert.domain + ' <span class="status-badge status-' + cert.status + '">' + cert.status + '</span></h3>';
                        html += '<p><strong>Provider:</strong> ' + cert.provider + '</p>';
                        html += '<p><strong>Created:</strong> ' + cert.created_at + '</p>';
                        if (cert.expires_at) {
                            html += '<p><strong>Expires:</strong> ' + cert.expires_at + '</p>';
                        }
                        
                        html += '<div class="certificate-actions">';
                        if (cert.status === 'issued') {
                            html += '<button class="button download-btn" data-cert-id="' + cert.id + '">Download</button>';
                            html += '<button class="button install-btn" data-cert-id="' + cert.id + '" data-domain="' + cert.domain + '">Install via cPanel</button>';
                        }
                        html += '</div>';
                        html += '</div><hr>';
                    });
                    $('#certificates-list').html(html);
                } else {
                    $('#certificates-list').html('<p>No certificates found.</p>');
                }
            },
            error: function() {
                $('#certificates-list').html('<p class="notice notice-error">Failed to load certificates.</p>');
            }
        });
    }
    
    // Refresh certificates
    $('#refresh-certificates').on('click', function() {
        loadCertificates();
    });
    
    // Download certificate
    $(document).on('click', '.download-btn', function() {
        var $btn = $(this);
        var certId = $btn.data('cert-id');
        
        if (!confirm('Download certificate in Nginx format?')) {
            return;
        }
        
        $btn.prop('disabled', true).text(wpSslManager.strings.downloading);
        
        $.ajax({
            url: wpSslManager.ajaxUrl,
            type: 'POST',
            data: {
                action: 'wp_ssl_download_certificate',
                nonce: wpSslManager.nonce,
                certificate_id: certId,
                format: 'nginx',
            },
            success: function(response) {
                if (response.success) {
                    alert('Certificate downloaded to: ' + response.data.path);
                } else {
                    alert('Error: ' + (response.data.message || wpSslManager.strings.error));
                }
                $btn.prop('disabled', false).text('Download');
            },
            error: function() {
                alert(wpSslManager.strings.error);
                $btn.prop('disabled', false).text('Download');
            }
        });
    });
    
    // Install certificate via cPanel
    $(document).on('click', '.install-btn', function() {
        var $btn = $(this);
        var certId = $btn.data('cert-id');
        var domain = $btn.data('domain');
        
        if (!confirm('Install SSL certificate via cPanel for ' + domain + '?')) {
            return;
        }
        
        $btn.prop('disabled', true).text(wpSslManager.strings.installing);
        
        $.ajax({
            url: wpSslManager.ajaxUrl,
            type: 'POST',
            data: {
                action: 'wp_ssl_install_certificate',
                nonce: wpSslManager.nonce,
                certificate_id: certId,
                domain: domain,
                format: 'nginx',
            },
            success: function(response) {
                if (response.success) {
                    alert('SSL certificate installed successfully!');
                } else {
                    alert('Error: ' + (response.data.message || wpSslManager.strings.error));
                }
                $btn.prop('disabled', false).text('Install via cPanel');
            },
            error: function() {
                alert(wpSslManager.strings.error);
                $btn.prop('disabled', false).text('Install via cPanel');
            }
        });
    });
    
    // Load certificates on page load if on certificates tab
    if ($('#certificates').hasClass('active')) {
        loadCertificates();
    }
});

